open Logarion
module FS = File_store
module A = Archive

let listing r order_opt reverse_opt number_opt paths_opt authors_opt topics_opt =
	let predicates = A.predicate A.authored authors_opt @ A.predicate A.topics topics_opt in
	let predicate text = List.fold_left (fun a e -> a && e text) true predicates in
	let list_text (t, fnames) = Printf.printf "%s %s %s 𐄁 %s%s\n"
		(Text.short_id t) Date.(pretty_date @@ listing t.Text.date)
		(Person.Set.to_string ~names_only:true t.Text.authors)
		t.Text.title (if paths_opt then (List.fold_left (Printf.sprintf "%s\n@ %s") "" fnames) else "")
	in
	match order_opt with
	| false -> FS.iter ~r ~predicate list_text
	| true ->
		let order = match reverse_opt with true -> FS.newest | false -> FS.oldest in
		match number_opt with
		| Some number -> FS.iter ~r ~predicate ~order ~number list_text
		| None -> FS.iter ~r ~predicate ~order list_text

open Cmdliner
let term =
	let recurse = Arg.(value & flag & info ["R"] ~doc:"recurse, include subdirs") in
	let reverse = Arg.(value & flag & info ["r"] ~doc:"reverse order") in
	let time   = Arg.(value & flag & info ["t"] ~doc:"sort by time, newest first") in
	let paths = Arg.(value & flag & info ["p"] ~doc:"show file paths") in
	let number = Arg.(value & opt (some int) None & info ["n"]
		~docv:"number" ~doc:"number of entries to list") in
	let authed = Arg.(value & opt (some string) None & info ["authored"]
		~docv:"comma-separated names" ~doc:"texts by authors") in
	let topics = Arg.(value & opt (some string) None & info ["topics"]
		~docv:"comma-separated topics" ~doc:"texts with topics") in
	Term.(const listing $ recurse $ time $ reverse $ number $ paths $ authed $ topics),
	Term.info "list" ~doc:"list texts" ~man:[ `S "DESCRIPTION";
		`P "List header information for current directory. If -R is used, list header
		information for texts found in subdirectories too, along with their filepaths" ]
