open Kosuzu
module FS = File_store
module A = Archive

let listing r order_opt reverse_opt number_opt paths_opt authors_opt topics_opt dir =
	let dir = if dir = "" then FS.txtdir () else dir in
	let predicates = A.predicate A.authored authors_opt @ A.predicate A.topics topics_opt in
	let predicate text = List.fold_left (fun a e -> a && e text) true predicates in
	let list_text (t, fnames) = Printf.printf "%s | %s | %s | %s %s\n"
		(Text.short_id t) Date.(pretty_date @@ listing t.Text.date)
		(Person.Set.to_string ~names_only:true t.Text.authors)
		t.Text.title (if paths_opt then (List.fold_left (Printf.sprintf "%s\n@ %s") "" fnames) else "")
	in
	match order_opt with
	| false -> FS.iter ~r ~dir ~predicate list_text
	| true ->
		let order = match reverse_opt with true -> FS.newest | false -> FS.oldest in
		match number_opt with
		| Some number -> FS.iter ~r ~dir ~predicate ~order ~number list_text
		| None -> FS.iter ~r ~dir ~predicate ~order list_text

open Cmdliner

let recurse = Arg.(value & flag & info ["R"] ~doc: "Recurse into subdirectories")
let reverse = Arg.(value & flag & info ["r"] ~doc: "Reverse order")
let time = Arg.(value & flag & info ["t"] ~doc: "Sort by time, newest first")
let paths = Arg.(value & flag & info ["p"] ~doc: "Show file paths")
let number = Arg.(value & opt (some int) None & info ["n"] ~docv: "number" ~doc: "Number of entries to list")
let authed = Arg.(value & opt (some string) None & info ["authored"] ~docv: "comma-separated names" ~doc: "Texts by authors")
let topics = Arg.(value & opt (some string) None & info ["topics"] ~docv: "comma-separated topics" ~doc: "Texts by topics")
let dir = Arg.(value & pos 0 string "" & info [] ~docv: "directory to index")

let listing_t = Term.(const listing $ recurse $ time $ reverse $ number $ paths $ authed $ topics $ dir)

let cmd =
  let doc = "List texts" in
  let man = [
      `S Manpage.s_description;
      `P "Displays text id, date, author, title for a directory.";
      `P "If directory argument is omitted, TXTDIR is used, where empty value defaults to ~/.local/share/texts.";
      `P "If -R is used, list header information for texts found in subdirectories, too." ]
  in
  let info = Cmd.info "list" ~doc ~man in
  Cmd.v info listing_t
