open Kosuzu

module Rel = struct

module Rel_set = Set.Make(String)
module Id_map = Map.Make(String)

type t = { last_rel: string; ref_set: String_set.t; rep_set: String_set.t }
type map_t = t Id_map.t

let empty = { last_rel = ""; ref_set = Rel_set.empty; rep_set = Rel_set.empty }
let empty_map = Id_map.empty

let acc_ref date source target = Id_map.update target (function
	| None -> Some { last_rel = date;
		ref_set = Rel_set.singleton source;
		rep_set = Rel_set.empty }
	| Some rel -> Some { rel with
		last_rel = if Date.compare date rel.last_rel > 0 then date else rel.last_rel;
		ref_set = Rel_set.add source rel.ref_set })

let acc_rep date source target = Id_map.update target (function
	| None -> Some { last_rel = date;
		rep_set = Rel_set.singleton source;
		ref_set = Rel_set.empty }
	| Some rel -> Some { rel with
		last_rel = if Date.compare date rel.last_rel > 0 then date else rel.last_rel;
		rep_set = Rel_set.add source rel.rep_set })

let acc_txt rels (text, _paths) =
	let acc_ref = acc_ref (Date.listing text.Text.date) text.Text.id in
	let acc_rep = acc_rep (Date.listing text.Text.date) text.Text.id in
	let rels = String_set.fold acc_ref (Text.set "references" text) rels in
	let rels = String_set.fold acc_rep (Text.set "in-reply-to" text) rels in
	rels

let acc_pck rels peer =
	let path = try List.hd peer.Peers.pack.Header_pack.info.locations with Failure _->"" in
	try Header_pack.fold
		(fun rels id t _title _authors _topics refs_ls reps_ls ->
			let acc_ref = acc_ref (Date.of_secs @@ Int32.to_int t) (Filename.concat path id) in
			let acc_rep = acc_rep (Date.of_secs @@ Int32.to_int t) (Filename.concat path id) in
			let rels = String_set.fold acc_ref (String_set.of_list refs_ls) rels in
			let rels = String_set.fold acc_rep (String_set.of_list reps_ls) rels in
			rels)
		rels peer.Peers.pack
	with e -> prerr_endline "acc_pck"; raise e
end


type t = {
	id: string;
	dir: string;
	kv: string Store.KV.t;
	topic_roots: string list;
	topics: (String_set.t * String_set.t) Topic_set.Map.t;
	relations: Rel.map_t;
	texts: Text.t list
}

type fn_t = {
	ext: string;
	page: (t -> Kosuzu.Text.t -> string) option;
	indices: (t -> unit) option;
}

let empty () = {
	id = ""; dir = "";
	kv = Store.KV.empty;
	topic_roots = [];
	topics = Topic_set.Map.empty;
	relations = Rel.Id_map.empty;
	texts = []
}
