let ext = ".atom"

let esc = Converter.Html.esc

let element tag content = "<" ^ tag ^ ">" ^ content ^ "</" ^ tag ^ ">"

let opt_element tag_name content =
  if content <> ""
  then element tag_name content
  else ""

module P = Parsers.Plain_text.Make (Converter.Html)

let id txt = "<id>urn:txtid:" ^ Kosuzu.(txt.Text.id) ^ "</id>\n"
let title text = "<title>" ^ esc text.Kosuzu.Text.title ^ "</title>\n"

let authors text =
	let u acc addr = acc ^ element "uri" addr in
  let open Kosuzu in
  let fn txt a =
  	a ^ "<author>" ^ (opt_element "name" @@ esc txt.Person.name)
  	^ (List.fold_left u "" txt.Person.addresses)
  	^ "</author>\n" in
  Person.Set.fold fn text.Text.authors ""

let updated txt = let open Kosuzu in
	"<updated>"^ Date.(txt.Text.date |> listing |> rfc_string) ^"</updated>\n"
	
let htm_entry base_url text =
  let open Kosuzu in
  let u = Text.short_id text in
  "<entry>\n<link rel=\"alternate\" href=\"" ^ base_url ^ "/" ^ u ^ ".htm\" />\n"
  ^ title text ^ id text ^ updated text ^ authors text
  ^ (opt_element "summary" @@ esc @@ Text.str "abstract" text)
  ^ String_set.fold (fun elt a -> a ^ "<category term=\"" ^ esc elt ^ "\"/>\n") (Text.set "topics" text) ""
  ^ "</entry>\n"

let gmi_entry base_url text =
  let open Kosuzu in
  let u = Text.short_id text in
  "<entry>\n<link rel=\"alternate\" href=\"" ^ base_url ^ "/" ^ u ^ ".gmi\" />\n"
  ^ title text ^ id text ^ updated text ^ authors text
  ^ (opt_element "summary" @@ esc @@ Text.str "abstract" text)
  ^ String_set.fold (fun elt a -> a ^ "<category term=\"" ^ elt ^ "\"/>\n") (Text.set "topics" text) ""
  ^ "</entry>\n"

let base_url kv protocol = try
	let locs = Kosuzu.Store.KV.find "Locations" kv in
	let _i = Str.(search_forward (regexp (protocol ^ "://[^;]*")) locs 0) in
	Str.(matched_string locs)
	with Not_found -> Printf.eprintf "Missing location for %s, add it to txt.conf\n" protocol; ""

let indices alternate_type c =
	let file name = Kosuzu.File_store.file (Filename.concat c.Conversion.dir name) in
	let title = try Kosuzu.Store.KV.find "Title" c.Conversion.kv with Not_found -> "" in
	let entry, fname, protocol_regexp = match alternate_type with
	| "text/gemini" -> gmi_entry, "gmi.atom", "gemini"
	| "text/html" | _ -> htm_entry, "feed.atom", "https?"
	in
	let base_url = base_url c.kv protocol_regexp in
	let self = Filename.concat base_url fname in
	file fname @@ (*TODO: alternate & self per url*)
	  {|<?xml version="1.0" encoding="utf-8"?><feed xmlns="http://www.w3.org/2005/Atom" xml:base="|} ^ base_url ^ {|"><title>|}
	  ^ title ^ {|</title><link rel="alternate" type="|} ^ alternate_type ^ {|" href="|}
	  ^ base_url ^ {|/" /><link rel="self" type="application/atom+xml" href="|}
	  ^ self ^ {|" /><id>urn:txtid:|} ^ c.Conversion.id ^ "</id><updated>"
	  ^ Kosuzu.Date.now () ^ "</updated>\n"
	  ^ List.fold_left (fun acc t -> acc ^ entry base_url t) "" c.texts
	  ^ "</feed>"

let converter format = Conversion.{ ext; page = None; indices = Some (indices format) }
