type t = Mustache.t

let of_string = Mustache.of_string
let of_file f = File.load f |> of_string

let string s = [Html.data s]
let section ~inverted:_ _name _contents = prerr_endline "Mustache sections unsupported"; []
let unescaped _elts = prerr_endline "Mustache unescaped not supported; used escaped instead"; []
let partial ?indent:_ _name _ _ = prerr_endline "Mustache sections unsupported"; []
let comment _ = [Html.data ""]
let concat = List.concat

let escaped_index ~from:_ ~n:_ _metas _e = [Html.data "temp"]
  (* match List.hd e with *)
  (* | "topics" -> *)
  (*    let topics = *)
  (*      ListLabels.fold_left *)
  (*        ~init:(Logarion.Meta.StringSet.empty) *)
  (*        ~f:(fun a e -> Logarion.Meta.unique_topics a e ) metas *)
  (*    in *)
  (*    Logarion.Meta.StringSet.fold (fun e a -> a ^ "<li><a href=\"/topic/" ^ e ^ "\">" ^ e ^ "</a></li>") topics "" *)

let header_custom template _linker archive =
  Mustache.fold ~string ~section ~escaped:(Html.Renderer.archive archive) ~unescaped ~partial ~comment ~concat template
  |> Html.header

let header_default linker archive =
  Html.(header [title [anchor (linker "/") [data archive.Logarion.Archive.Configuration.title]]])

let meta meta =
  let open Logarion.Meta in
  let abstract = meta.abstract in
  let authors = List.map (fun elt -> elt.Author.name) @@ AuthorSet.elements meta.authors in
  let date = Date.(pretty_date @@ listing meta.date) in
  let series = stringset_csv meta.series in
  let topics = stringset_csv meta.topics in
  let keywords = stringset_csv meta.keywords in
  let uuid = Id.to_string meta.uuid in
  Html.meta ~abstract ~authors ~date ~series ~topics ~keywords ~uuid

let body_custom template note =
  Mustache.fold ~string ~section ~escaped:(Html.Renderer.note note) ~unescaped ~partial ~comment ~concat template
  |> Html.note

let body_default note =
  Html.note
    [ Html.title [Html.unescaped_data note.Logarion.Note.meta.Logarion.Meta.title]; (* Don't add title if body contains one *)
      meta note.meta;
      Html.unescaped_data @@ Omd.to_html @@ Omd.of_string note.Logarion.Note.body ]

let page ~style linker title header body =
  Html.to_string @@ Html.page ~style linker title header body
                
let of_config config k = match config with
  | Error msg -> prerr_endline ("Couldn't load [templates] section;" ^ msg); None
  | Ok c ->
     let open Confix.ConfixToml in
     path c ("templates" / k)

let converter default custom = function
  | Some p ->
     if Confix.Config.Path.path_exists p then custom @@ of_file p
     else (prerr_endline @@ "Couldn't find: " ^ Fpath.to_string p; default)
  | None -> default
          
let header_converter config = converter header_default header_custom @@ of_config config "header"
let body_converter   config = converter body_default   body_custom   @@ of_config config "body"

let default_style = Html.default_style

let page_of_index ~style linker header archive metas =
  page ~style linker ("Index | " ^ archive.Logarion.Archive.Configuration.title) (header linker archive) (Html.main (Html.listing_index "" metas))

let page_of_log ~style linker header archive metas =
  page ~style linker ("Log | " ^ archive.Logarion.Archive.Configuration.title) (header linker archive) (Html.main [Html.listing_texts "" metas])

let page_of_note ~style linker header body archive note =
  page ~style linker note.Logarion.Note.meta.Logarion.Meta.title (header linker archive) (body note)

let page_of_msg ~style linker header archive title msg =
  page ~style linker title (header linker archive) (Html.div [Html.data msg])
